/* SPDX-License-Identifier: GPL-3.0-or-later WITH cryptsetup-OpenSSL-exception */
/*
    This file is part of sscg.

    sscg is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    sscg is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with sscg.  If not, see <http://www.gnu.org/licenses/>.

    In addition, as a special exception, the copyright holders give
    permission to link the code of portions of this program with the
    OpenSSL library under certain conditions as described in each
    individual source file, and distribute linked combinations
    including the two.
    You must obey the GNU General Public License in all respects
    for all of the code used other than OpenSSL.  If you modify
    file(s) with this exception, you may extend this exception to your
    version of the file(s), but you are not obligated to do so.  If you
    do not wish to do so, delete this exception statement from your
    version.  If you delete this exception statement from all source
    files in the program, then also delete it here.

    Copyright 2017-2025 by Stephen Gallagher <sgallagh@redhat.com>
*/

/* This is a master header file that should be included by all
   sscg source files. */


#ifndef _SSCG_H
#define _SSCG_H

#include <errno.h>
#include <openssl/bn.h>
#include <openssl/ssl.h>
#include <openssl/err.h>
#include <openssl/ui.h>
#include <talloc.h>
#include <stdint.h>

#include "include/logging.h"

#ifndef _
#ifdef HAVE_GETTEXT
#define _(STRING) gettext (STRING)
#else
#define _(STRING) STRING
#endif /* HAVE_GETTEXT */
#endif /* _ */

#ifndef EOK
#define EOK 0
#endif

#ifndef discard_const
#define discard_const(ptr) ((void *)((uintptr_t)(ptr)))
#endif

#ifndef talloc_zfree
#define talloc_zfree(ptr)                                                     \
  do                                                                          \
    {                                                                         \
      talloc_free (discard_const (ptr));                                      \
      ptr = NULL;                                                             \
    }                                                                         \
  while (0)
#endif

#define CHECK_MEM(ptr)                                                        \
  do                                                                          \
    {                                                                         \
      if (!ptr)                                                               \
        {                                                                     \
          ret = ENOMEM;                                                       \
          goto done;                                                          \
        }                                                                     \
    }                                                                         \
  while (0)

#define CHECK_OK(_ret)                                                        \
  do                                                                          \
    {                                                                         \
      if (_ret != EOK)                                                        \
        {                                                                     \
          goto done;                                                          \
        }                                                                     \
    }                                                                         \
  while (0)

/* The function changed in 1.1, but the library and reason names did not */
#ifndef UI_F_UI_SET_RESULT_EX
#define UI_F_UI_SET_RESULT_EX UI_F_UI_SET_RESULT
#endif

#define CHECK_SSL(_sslret, _fn)                                               \
  do                                                                          \
    {                                                                         \
      if (_sslret != 1)                                                       \
        {                                                                     \
          /* Get information about error from OpenSSL */                      \
          unsigned long _ssl_error = ERR_get_error ();                        \
          if ((ERR_GET_LIB (_ssl_error) == ERR_LIB_UI)                        \
              && ((ERR_GET_REASON (_ssl_error) == UI_R_RESULT_TOO_LARGE)      \
                  || (ERR_GET_REASON (_ssl_error) == UI_R_RESULT_TOO_SMALL))) \
            {                                                                 \
              fprintf (                                                       \
                stderr,                                                       \
                "Passphrases must be between %d and %d characters. \n",       \
                SSCG_MIN_KEY_PASS_LEN,                                        \
                SSCG_MAX_KEY_PASS_LEN);                                       \
              ret = EINVAL;                                                   \
              goto done;                                                      \
            }                                                                 \
          fprintf (stderr,                                                    \
                   "Error occurred in " #_fn ": [%s].\n",                     \
                   ERR_error_string (_ssl_error, NULL));                      \
          ret = EIO;                                                          \
          goto done;                                                          \
        }                                                                     \
    }                                                                         \
  while (0)

struct sscg_options
{
  /* How noisy to be when printing information */
  enum sscg_verbosity verbosity;

  /* Whether to print the version and exit */
  int print_version;

  /* How long should certificates be valid (in days) */
  int lifetime;

  /* Subject information */
  const char *country;
  const char *state;
  const char *locality;
  const char *org;
  const char *org_unit;
  const char *email;
  const char *hostname;
  char **subject_alt_names;

  /* Encryption requirements */
  int system_security_level;
  char *key_type;
  int rsa_key_strength;
  int minimum_rsa_key_strength;
  int mldsa_nist_level;
  char *ec_curve;
  char *hash_alg;
  char *cipher_alg;
  const EVP_CIPHER *cipher;
  const EVP_MD *hash_fn;

  int ca_key_pass_prompt;
  int cert_key_pass_prompt;
  int client_key_pass_prompt;
  int skip_dhparams;

  /* Output Files */
  struct sscg_stream **streams;

  char *ca_file;
  char *ca_key_file;
  int ca_mode;
  int ca_key_mode;
  char *ca_key_password;
  char *ca_key_passfile;

  char *cert_file;
  char *cert_key_file;
  int cert_mode;
  int cert_key_mode;
  char *cert_key_password;
  char *cert_key_passfile;

  char *client_file;
  char *client_key_file;
  int client_mode;
  int client_key_mode;
  char *client_key_password;
  char *client_key_passfile;

  char *crl_file;
  int crl_mode;

  char *dhparams_file;
  int dhparams_mode;

  /* Diffie-Hellman Parameters */
  char *dhparams_group;
  int dhparams_prime_len;
  int dhparams_generator;

  /* Overwrite the output files */
  int overwrite;
};


enum sscg_cert_type
{
  SSCG_CERT_TYPE_UNKNOWN = -1,
  SSCG_CERT_TYPE_SERVER,
  SSCG_CERT_TYPE_CLIENT,

  SSCG_NUM_CERT_TYPES
};

#define SSCG_MIN_KEY_PASS_LEN 4
#define SSCG_MAX_KEY_PASS_LEN 1023

/* RFC 1035, section 2.3.4 (Size Limits) */
#define MAX_HOST_LEN 63
#define MAX_FQDN_LEN 255

int
sscg_handle_arguments (TALLOC_CTX *mem_ctx,
                       int argc,
                       const char **argv,
                       struct sscg_options **config);

#endif /* _SSCG_H */
