/*************************************************************************
* Copyright (C) 2014 Intel Corporation
*
* Licensed under the Apache License,  Version 2.0 (the "License");
* you may not use this file except in compliance with the License.
* You may obtain a copy of the License at
*
* 	http://www.apache.org/licenses/LICENSE-2.0
*
* Unless required by applicable law  or agreed  to  in  writing,  software
* distributed under  the License  is  distributed  on  an  "AS IS"  BASIS,
* WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
* See the License for the  specific  language  governing  permissions  and
* limitations under the License.
*************************************************************************/

/*
//
//  Purpose:
//     Cryptography Primitive.
//     HMAC General Functionality
//
//  Contents:
//        ippsHMAC_Init()
//
*/

#include "owndefs.h"
#include "owncp.h"
#include "pcphmac.h"
#include "pcptool.h"

/*F*
//    Name: ippsHMAC_Init
//
// Purpose: Init HMAC state.
//
// Returns:                Reason:
//    ippStsNullPtrErr           pKey == NULL
//                               pState == NULL
//    ippStsLengthErr            keyLen <0
//    ippStsNotSupportedModeErr  if algID is not match to supported hash alg
//    ippStsNoErr                no errors
//
// Parameters:
//    pKey        pointer to the secret key
//    keyLen      length (bytes) of the secret key
//    pState      pointer to the HMAC state
//    hashAlg     hash alg ID
//
*F*/

/* clang-format off */
IPPFUN(IppStatus, ippsHMAC_Init, (const Ipp8u* pKey,
                                  int keyLen,
                                  IppsHMACState* pCtx,
                                  IppHashAlgId hashAlg))
/* clang-format on */
{
    //int mbs;

    /* get algorithm id */
    hashAlg = cpValidHashAlg(hashAlg);
    /* test hash alg */
    IPP_BADARG_RET(ippHashAlg_Unknown == hashAlg, ippStsNotSupportedModeErr);
    /* check if the algorithm is from the sha3 family (SHA3 is not supported in non-rmf methods)*/
    IPP_BADARG_RET(cpIsSHA3AlgID(hashAlg), ippStsNotSupportedModeErr);

    /* test pState pointer */
    IPP_BAD_PTR1_RET(pCtx);

    /* test key pointer and key length */
    IPP_BAD_PTR1_RET(pKey);
    IPP_BADARG_RET(0 > keyLen, ippStsLengthErr);

    /* set state ID */
    HMAC_SET_CTX_ID(pCtx);

    /* init hash context */
    ippsHashInit(&HASH_CTX(pCtx), hashAlg);

    {
        int n;

        /* hash specific */
        IppsHashState* pHashCtx = &HASH_CTX(pCtx);
        int mbs                 = cpHashMBS(hashAlg);
        int hashSize            = cpHashSize(hashAlg);

        /* copyMask = keyLen>mbs? 0xFF : 0x00 */
        int copyMask = (mbs - keyLen) >> (BITSIZE(int) - 1);

        /* actualKeyLen = keyLen>mbs? hashSize:keyLen */
        int actualKeyLen = (hashSize & copyMask) | (keyLen & ~copyMask);

        /* compute hash(key, keyLen) just in case */
        ippsHashUpdate(pKey, keyLen, pHashCtx);
        ippsHashFinal(HASH_BUFF(pHashCtx), pHashCtx);

        /* copy either key or hash(key) into ipad- and opad- buffers */
        MASKED_COPY_BNU(pCtx->ipadKey, (Ipp8u)copyMask, HASH_BUFF(pHashCtx), pKey, actualKeyLen);
        MASKED_COPY_BNU(pCtx->opadKey, (Ipp8u)copyMask, HASH_BUFF(pHashCtx), pKey, actualKeyLen);

        /* XOR-ing key */
        for (n = 0; n < actualKeyLen; n++) {
            pCtx->ipadKey[n] ^= (Ipp8u)IPAD;
            pCtx->opadKey[n] ^= (Ipp8u)OPAD;
        }
        for (; n < mbs; n++) {
            pCtx->ipadKey[n] = (Ipp8u)IPAD;
            pCtx->opadKey[n] = (Ipp8u)OPAD;
        }

        /* ipad key processing */
        ippsHashUpdate(pCtx->ipadKey, mbs, pHashCtx);

        return ippStsNoErr;
    }
}
