/*
 *  $Id: app-graph-window.c 29094 2026-01-06 17:53:17Z yeti-dn $
 *  Copyright (C) 2025 David Necas (Yeti)
 *  E-mail: yeti@gwyddion.net
 *
 *  This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 *  License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any
 *  later version.
 *
 *  This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 *  warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
 *  details.
 *
 *  You should have received a copy of the GNU General Public License along with this program; if not, write to the
 *  Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#define DEBUG 1
#include "config.h"
#include <glib/gi18n-lib.h>
#include <gtk/gtk.h>

#include "libgwyddion/macros.h"
#include "libgwyui/utils.h"

#include "libgwyapp/gwyapp.h"
#include "libgwyapp/sanity.h"
#include "libgwyapp/gwyappinternal.h"

struct _GwyAppGraphWindowPrivate {
    GtkWidget *menubar;
    GwySensitivityGroup *sens_group;

    GtkWidget *meta_widget;
    gulong meta_destroyed_id;

    GObject *object;
    gulong object_notify_id;

    GtkWidget *log_widget;
    gulong log_destroyed_id;

    GwyFile *file;
    gulong item_changed_id;
    GwyDataKind data_kind;
    gint id;
};

static void       finalize           (GObject *object);
static void       dispose            (GObject *object);
static gboolean   configured         (GtkWidget *widget,
                                      GdkEventConfigure *event);
static void       size_allocated     (GtkWidget *widget,
                                      GdkRectangle *allocation);
static void       save_size          (GwyAppGraphWindow *window,
                                      const GdkRectangle *allocation);
static void       add_menubar        (GwyAppGraphWindow *window,
                                      GtkBox *vbox);
static GtkWidget* create_view_menu   (GwyAppGraphWindow *window,
                                      GtkAccelGroup *accel_group);
static void       show_meta_browser  (GwyAppGraphWindow *window);
static void       meta_destroyed     (GwyAppGraphWindow *window);
static void       show_log_browser   (GwyAppGraphWindow *window);
static void       log_destroyed      (GwyAppGraphWindow *window);
static void       item_changed       (GwyContainer *container,
                                      GQuark key,
                                      GwyAppGraphWindow *window);
static void       object_notify      (GObject *object,
                                      const GParamSpec *pspec,
                                      GwyAppGraphWindow *window);
static void       update_sensitivity (GwyAppGraphWindow *window);
static void       initial_setup      (GwyAppGraphWindow *window);
static void       update_window_title(GwyAppGraphWindow *windiw);

static GtkWindowClass *parent_class = NULL;

G_DEFINE_TYPE_WITH_CODE(GwyAppGraphWindow, gwy_app_graph_window, GWY_TYPE_GRAPH_WINDOW,
                        G_ADD_PRIVATE(GwyAppGraphWindow))

static void
gwy_app_graph_window_class_init(GwyAppGraphWindowClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS(klass);
    GtkWidgetClass *widget_class = GTK_WIDGET_CLASS(klass);

    parent_class = gwy_app_graph_window_parent_class;

    gobject_class->finalize = finalize;
    gobject_class->dispose = dispose;

    widget_class->configure_event = configured;
    widget_class->size_allocate = size_allocated;
    /* TODO: Various button/key press events, popup-menu event (which we might not need if we have a menubar). */
}

static void
dispose(GObject *object)
{
    GwyAppGraphWindow *window = GWY_APP_GRAPH_WINDOW(object);
    GwyAppGraphWindowPrivate *priv = window->priv;

    gwy_set_member_object(window, NULL, GWY_TYPE_GRAPH_MODEL, &priv->object,
                          "notify::n-curves", G_CALLBACK(object_notify), &priv->object_notify_id, 0,
                          NULL);
    g_clear_signal_handler(&priv->item_changed_id, priv->file);

    G_OBJECT_CLASS(parent_class)->dispose(object);
}

static void
finalize(GObject *object)
{
    GwyAppGraphWindow *window = GWY_APP_GRAPH_WINDOW(object);
    GwyAppGraphWindowPrivate *priv = window->priv;

    g_clear_object(&priv->sens_group);

    G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
gwy_app_graph_window_init(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv;

    window->priv = priv = gwy_app_graph_window_get_instance_private(window);

    priv->sens_group = gwy_sensitivity_group_new();
    gwy_app_add_main_accel_group(GTK_WINDOW(window));

    GtkWidget *curves = gwy_graph_window_get_graph_curves(GWY_GRAPH_WINDOW(window));
    GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(curves));
    gwy_app_set_tree_model_kind(model, "graph-curves");
}

/**
 * gwy_app_graph_window_new:
 * @file: (transfer none): A data file container.
 * @data_kind: Type of the data object.
 * @id: Data item id.
 *
 * Creates a new app image data window.
 *
 * The arguments must correspond to a valid data object in @file which can be visualised as a graph. Possible data
 * kinds currently include only @GWY_FILE_GRAPH.
 *
 * Returns: (transfer full) (constructor): A newly created window.
 **/
GtkWindow*
gwy_app_graph_window_new(GwyFile *file,
                         GwyDataKind data_kind,
                         gint id)
{
    static const GwyEnum help_sections[] = {
        { "graph-windows", GWY_FILE_GRAPH, },
    };

    GwyAppGraphWindow *window = g_object_new(GWY_TYPE_APP_GRAPH_WINDOW, NULL);
    GwyAppGraphWindowPrivate *priv = window->priv;

    g_assert(GWY_IS_FILE(file));
    g_assert(data_kind == GWY_FILE_GRAPH);
    g_assert(id >= 0);
    priv->file = file;
    priv->data_kind = data_kind;
    priv->id = id;

    const gchar *helploc = gwy_enum_to_string(data_kind, help_sections, G_N_ELEMENTS(help_sections));
    if (helploc)
        gwy_help_add_to_window(GTK_WINDOW(window), helploc, NULL, GWY_HELP_DEFAULT);

    GtkBox *vbox = GTK_BOX(gtk_bin_get_child(GTK_BIN(window)));
    g_assert(GTK_IS_BOX(vbox));
    add_menubar(window, vbox);

    /* TODO: There are various popups and stuff. */

    initial_setup(window);
    priv->item_changed_id = g_signal_connect(file, "item-changed", G_CALLBACK(item_changed), window);

    return (GtkWindow*)window;
}

/**
 * gwy_app_graph_window_get_file:
 * @window: App image data window.
 *
 * Gets the data file containing data displayed in an app image data window.
 *
 * Returns: (transfer none): The file data container.
 **/
GwyFile*
gwy_app_graph_window_get_file(GwyAppGraphWindow *window)
{
    g_return_val_if_fail(GWY_IS_APP_GRAPH_WINDOW(window), NULL);
    return window->priv->file;
}

/**
 * gwy_app_graph_window_get_data_kind:
 * @window: App image data window.
 *
 * Gets the kind of data displayed in an app image data window.
 *
 * Returns: The data kind.
 **/
GwyDataKind
gwy_app_graph_window_get_data_kind(GwyAppGraphWindow *window)
{
    g_return_val_if_fail(GWY_IS_APP_GRAPH_WINDOW(window), GWY_FILE_NONE);
    return window->priv->data_kind;
}

/**
 * gwy_app_graph_window_get_id:
 * @window: App image data window.
 *
 * Gets the numerical id of data item displayed in an app image data window.
 *
 * Returns: The numerical id.
 **/
gint
gwy_app_graph_window_get_id(GwyAppGraphWindow *window)
{
    g_return_val_if_fail(GWY_IS_APP_GRAPH_WINDOW(window), -1);
    return window->priv->id;
}

static void
item_changed(GwyContainer *container, GQuark key, GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    GwyFileKeyParsed parsed;

    const gchar *strkey = g_quark_to_string(key);
    gwy_debug("<%s>", strkey);
    gboolean parsed_ok = gwy_file_parse_key(strkey, &parsed);
    g_return_if_fail(parsed_ok);

    GwyDataKind data_kind = parsed.data_kind;
    gint id = parsed.id;
    GwyFilePiece piece = parsed.piece;
    if (data_kind == GWY_FILE_NONE && piece == GWY_FILE_PIECE_FILENAME) {
        update_window_title(window);
        return;
    }

    /* FIXME: This is simplistic. */
    if (data_kind != priv->data_kind || id != priv->id) {
        return;
    }

    GwyFile *file = GWY_FILE(container);
    GwyGraphWindow *datawindow = GWY_GRAPH_WINDOW(window);
    GwyGraph *graph = GWY_GRAPH(gwy_graph_window_get_graph(datawindow));
    if (data_kind == GWY_FILE_GRAPH && piece == GWY_FILE_PIECE_NONE) {
        GwyGraphModel *gmodel = gwy_file_get_graph(file, id);
        gwy_graph_set_model(graph, gmodel);
        gwy_set_member_object(window, gmodel, GWY_TYPE_GRAPH_MODEL, &priv->object,
                              "notify::n-curves", G_CALLBACK(object_notify), &priv->object_notify_id, 0,
                              NULL);

        GwyMenuSensFlags sensflags = GWY_MENU_FLAG_GRAPH;
        gwy_sensitivity_group_set_state(priv->sens_group, sensflags, priv->object ? sensflags : 0);
    }
}

static void
object_notify(G_GNUC_UNUSED GObject *object,
              G_GNUC_UNUSED const GParamSpec *pspec,
              GwyAppGraphWindow *window)
{
    update_sensitivity(window);
}

static void
update_sensitivity(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    GwyMenuSensFlags sensflags = GWY_MENU_FLAG_GRAPH_CURVE;
    GObject *object = priv->object;
    gint ncurves = object ? gwy_graph_model_get_n_curves(GWY_GRAPH_MODEL(object)) : 0;
    gwy_sensitivity_group_set_state(priv->sens_group, sensflags, ncurves ? sensflags : 0);
}

static void
initial_setup(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    GwyDataKind data_kind = priv->data_kind;
    gint id = priv->id;
    GwyFileKeyParsed parsed = { .data_kind = data_kind, .id = id, .piece = GWY_FILE_PIECE_NONE, .suffix = NULL };
    GwyContainer *container = GWY_CONTAINER(priv->file);

    item_changed(container, gwy_file_form_key(&parsed), window);
    parsed.piece = GWY_FILE_PIECE_VIEW;
    update_window_title(window);
    update_sensitivity(window);
    _gwy_restore_window_size(GTK_WINDOW(window), container, gwy_file_form_string_key(&parsed));
}

static void
add_menubar(GwyAppGraphWindow *window, GtkBox *vbox)
{
    GwyAppGraphWindowPrivate *priv = window->priv;

    GtkAccelGroup *accel_group = NULL;
    GtkWidget *main_window = gwy_app_main_window_get();
    if (main_window)
        accel_group = GTK_ACCEL_GROUP(g_object_get_data(G_OBJECT(main_window), "accel_group"));

    GwyDataKind data_kind = priv->data_kind;
    GtkWidget *funcmenu = NULL, *item;
    const gchar *label = NULL;
    if (data_kind == GWY_FILE_GRAPH) {
        funcmenu = gwy_app_graph_menu(accel_group, priv->sens_group);
        label = _("_Graph");
    }

    priv->menubar = gtk_menu_bar_new();
    gtk_box_pack_start(vbox, priv->menubar, FALSE, FALSE, 0);
    gtk_box_reorder_child(vbox, priv->menubar, 0);

    if (funcmenu) {
        item = gtk_menu_item_new_with_mnemonic(label);
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), funcmenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(priv->menubar), item);
    }

    if (funcmenu) {
        GtkWidget *recentmenu = gwy_app_create_recent_func_menu(data_kind, GTK_WINDOW(window), priv->sens_group);
        item = gtk_menu_item_new_with_mnemonic(("_Recent"));
        gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), recentmenu);
        gtk_menu_shell_append(GTK_MENU_SHELL(priv->menubar), item);
    }

    item = gtk_menu_item_new_with_mnemonic(("Vie_w"));
    gtk_menu_item_set_submenu(GTK_MENU_ITEM(item), create_view_menu(window, accel_group));
    gtk_menu_shell_append(GTK_MENU_SHELL(priv->menubar), item);
}

/* FIXME: This should probably be a small toolbar, which could include the ‘Change Preview’ functions for volume
 * and curve map data. */
static GtkWidget*
create_view_menu(GwyAppGraphWindow *window, GtkAccelGroup *accel_group)
{
    GtkAccelFlags accel_flags = GTK_ACCEL_VISIBLE | GTK_ACCEL_LOCKED;
    GtkWidget *item;

    GtkWidget *menu = gtk_menu_new();
    GtkMenuShell *shell = GTK_MENU_SHELL(menu);
    if (accel_group)
        gtk_menu_set_accel_group(GTK_MENU(menu), accel_group);

    item = gtk_menu_item_new_with_mnemonic(_("_Metadata..."));
    gtk_widget_add_accelerator(item, "activate", accel_group,
                               GDK_KEY_B, GDK_CONTROL_MASK | GDK_SHIFT_MASK, accel_flags);
    gtk_menu_shell_append(shell, item);
    g_signal_connect_swapped(item, "activate", G_CALLBACK(show_meta_browser), window);

    item = gtk_menu_item_new_with_mnemonic(_("_Log..."));
    gtk_menu_shell_append(shell, item);
    g_signal_connect_swapped(item, "activate", G_CALLBACK(show_log_browser), window);

    return menu;
}

static void
show_meta_browser(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    if (!priv->meta_widget) {
        priv->meta_widget = gwy_meta_browser_new(priv->file, priv->data_kind, priv->id);
        priv->meta_destroyed_id = g_signal_connect_swapped(priv->meta_widget, "destroy",
                                                           G_CALLBACK(meta_destroyed), window);
    }
    gtk_window_present(GTK_WINDOW(priv->meta_widget));
}

static void
meta_destroyed(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    g_clear_signal_handler(&priv->meta_destroyed_id, priv->meta_widget);
    priv->meta_widget = NULL;
}

static void
show_log_browser(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    if (!priv->log_widget) {
        priv->log_widget = gwy_log_browser_new(priv->file, priv->data_kind, priv->id);
        priv->log_destroyed_id = g_signal_connect_swapped(priv->log_widget, "destroy",
                                                          G_CALLBACK(log_destroyed), window);
    }
    gtk_window_present(GTK_WINDOW(priv->log_widget));
}

static void
log_destroyed(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    g_clear_signal_handler(&priv->log_destroyed_id, priv->log_widget);
    priv->log_widget = NULL;
}

static void
update_window_title(GwyAppGraphWindow *window)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    gchar *basename;
    const gchar *filename;
    if (gwy_container_gis_string(GWY_CONTAINER(priv->file), gwy_file_key_filename(), &filename))
        basename = g_path_get_basename(filename);
    else
        basename = g_strdup(_("Untitled"));

    gwy_graph_window_set_data_name(GWY_GRAPH_WINDOW(window), basename);
    g_free(basename);
}

static gboolean
configured(GtkWidget *widget, GdkEventConfigure *event)
{
    GdkRectangle allocation = { .x = event->x, .y = event->y, .width = event->width, .height = event->height };
    save_size(GWY_APP_GRAPH_WINDOW(widget), &allocation);
    return GTK_WIDGET_CLASS(parent_class)->configure_event(widget, event);
}

static void
size_allocated(GtkWidget *widget, GdkRectangle *allocation)
{
    save_size(GWY_APP_GRAPH_WINDOW(widget), allocation);
    GTK_WIDGET_CLASS(parent_class)->size_allocate(widget, allocation);
}

static void
save_size(GwyAppGraphWindow *window, const GdkRectangle *allocation)
{
    GwyAppGraphWindowPrivate *priv = window->priv;
    /* Someone created us with plain g_object_new(). */
    g_return_if_fail(priv->file);
    GtkWidget *widget = gwy_graph_window_get_graph(GWY_GRAPH_WINDOW(window));
    _gwy_save_widget_screen_size(widget, allocation, priv->file, priv->data_kind, priv->id, TRUE);
}

/* FIXME: Where should it go? */
/* Use generic arguments and low-level key manipulation to make it useful with weird things like 3D views, program
 * windows with size in settings (not in the file), etc. */
void
_gwy_restore_window_size(GtkWindow *window,
                         GwyContainer *container, const gchar *prefix)
{
    if (!container || !prefix)
        return;

    gsize len = strlen(prefix);
    gchar key[len + 16];
    memcpy(key, prefix, len);

    gdouble relsize;
    strcpy(key + len, "/relative-size");
    if (!gwy_container_gis_double_by_name(container, key, &relsize))
        return;

    gint w, h;
    strcpy(key + len, "/width");
    if (!gwy_container_gis_int32_by_name(container, key, &w))
        return;
    strcpy(key + len, "/height");
    if (!gwy_container_gis_int32_by_name(container, key, &h))
        return;

    if (w <= 0 || h <= 0 || relsize <= 0.0)
        return;

    gdouble scw = gwy_get_screen_width(GTK_WIDGET(window));
    gdouble sch = gwy_get_screen_height(GTK_WIDGET(window));
    gdouble newrelsize = MAX(w/scw, h/sch);
    gwy_debug("restoring generic window size: relsize %g, size %dx%d, newrelsize %g",
              relsize, w, h, newrelsize);

    /* If the window will be small we can just apply the saved zoom.  Should it be larger though, we must check if it
     * is not too large and better show it at defaut size than huge. */
    if (newrelsize > 1.2*relsize || newrelsize > 0.9)
        return;
    gtk_window_set_default_size(window, w, h);
}

/**
 * SECTION: app-image-window
 * @title: GwyAppGraphWindow
 * @short_description: Windows for data shown as images
 *
 * #GwyAppGraphWindow wraps #GwyGraphWindow and updates it to changes in the file.
 **/

/* vim: set cin columns=120 tw=118 et ts=4 sw=4 cino=>1s,e0,n0,f0,{0,}0,^0,\:1s,=0,g1s,h0,t0,+1s,c3,(0,u0 : */
