"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VdmxTableIo = exports.VdmxGroup = exports.VdmxRatioRange = void 0;
const bin_util_1 = require("@ot-builder/bin-util");
const ot_metadata_1 = require("@ot-builder/ot-metadata");
const primitive_1 = require("@ot-builder/primitive");
exports.VdmxRatioRange = {
    read(bp) {
        return {
            bCharSet: bp.uint8(),
            xRatio: bp.uint8(),
            yStartRatio: bp.uint8(),
            yEndRatio: bp.uint8()
        };
    },
    write(b, r) {
        b.uint8(r.bCharSet);
        b.uint8(r.xRatio);
        b.uint8(r.yStartRatio);
        b.uint8(r.yEndRatio);
    }
};
exports.VdmxGroup = {
    read(bp) {
        const entries = new Map();
        const recs = bp.uint16();
        bp.uint8(); // skip `startsz`
        bp.uint8(); // skip `endsz`
        for (let recIndex = 0; recIndex < recs; recIndex++) {
            const yPelHeight = bp.uint16();
            entries.set(yPelHeight, {
                yMax: bp.int16(),
                yMin: bp.int16()
            });
        }
        return entries;
    },
    write(b, entries) {
        const recs = entries.size;
        const sorted = [...entries].sort((a, b) => a[0] - b[0]);
        const startsz = recs ? sorted[0][0] : 0;
        const endsz = recs ? sorted[recs - 1][0] : 0;
        b.uint16(recs);
        b.uint8(startsz);
        b.uint8(endsz);
        for (const entry of sorted) {
            b.uint16(entry[0]); // yPelHeight
            b.int16(entry[1].yMax);
            b.int16(entry[1].yMin);
        }
    }
};
exports.VdmxTableIo = {
    read(view) {
        const version = view.uint16();
        const table = new ot_metadata_1.Vdmx.Table(version);
        view.uint16(); // skip `numRecs`
        const numRatios = view.uint16();
        for (let ratioIndex = 0; ratioIndex < numRatios; ratioIndex++) {
            const ratio = view.next(exports.VdmxRatioRange);
            const group = new ot_metadata_1.Vdmx.VdmxRecord();
            group.ratioRange = ratio;
            table.records.push(group);
        }
        for (let groupIndex = 0; groupIndex < numRatios; groupIndex++) {
            const entries = view.ptr16().next(exports.VdmxGroup);
            table.records[groupIndex].entries = entries;
        }
        return table;
    },
    write(frag, table) {
        const numRatios = table.records.length;
        frag.uint16(table.version);
        const numRecsReserve = frag.reserve(primitive_1.UInt16);
        frag.uint16(numRatios);
        const groups = new Array();
        for (const record of table.records) {
            frag.push(exports.VdmxRatioRange, record.ratioRange);
            groups.push(bin_util_1.Frag.from(exports.VdmxGroup, record.entries));
        }
        const headerSize = frag.size + primitive_1.UInt16.size * numRatios;
        const groupPack = bin_util_1.Frag.packMany(groups);
        for (const offset of groupPack.rootOffsets)
            frag.uint16(offset + headerSize);
        frag.bytes(groupPack.buffer);
        const numRecs = new Set(groupPack.rootOffsets).size;
        numRecsReserve.fill(numRecs);
    }
};
//# sourceMappingURL=index.js.map