# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""Unit tests for SystemImageBuild class."""

from typing import Any

import debusine.worker.tags as wtags
from debusine.tasks.models import BaseDynamicTaskData, DebootstrapVariant
from debusine.tasks.systemimagebuild import SystemImageBuild
from debusine.tasks.tests.helper_mixin import (
    ExternalTaskHelperMixin,
    MockTaskDatabase,
)
from debusine.test import TestCase


class SystemImageBuildImpl(SystemImageBuild):
    """Implementation of SystemImageBuild ontology."""

    def _cmdline(self) -> list[str]:
        return []  # pragma: no cover


class SystemImageBuildTests(
    ExternalTaskHelperMixin[SystemImageBuildImpl], TestCase
):
    """Tests SystemImageBuild class."""

    SAMPLE_TASK_DATA: dict[str, Any] = {
        "bootstrap_options": {
            "architecture": "amd64",
            "extra_packages": ["hello", "python3"],
            "use_signed_by": True,
        },
        "bootstrap_repositories": [
            {
                "mirror": "https://deb.debian.org/debian",
                "suite": "stable",
                "components": ["main", "contrib"],
                "check_signature_with": "system",
            },
            {
                "types": ["deb-src"],
                "mirror": "https://example.com",
                "suite": "bullseye",
                "components": ["main"],
                "check_signature_with": "system",
                "keyring": {"url": "https://example.com/keyring.gpg"},
            },
        ],
        "disk_image": {
            "format": "raw",
            "partitions": [
                {
                    "size": 2,
                    "filesystem": "ext4",
                },
            ],
        },
    }

    def setUp(self) -> None:
        """Initialize test."""
        super().setUp()
        self.configure_task()

    def configure_task(
        self,
        task_data: dict[str, Any] | None = None,
        override: dict[str, Any] | None = None,
        remove: list[str] | None = None,
    ) -> None:
        """Perform further setup."""
        super().configure_task(task_data, override, remove)
        self.task_db = MockTaskDatabase()

    def test_get_provided_worker_tags(self) -> None:
        self.assertCountEqual(
            self.task.get_provided_worker_tags(),
            [
                wtags.TASK_PREFIX + "worker:systemimagebuildimpl:version:1",
            ],
        )

    def test_compute_system_required_tags(self) -> None:
        self.assertCountEqual(self.task.compute_system_required_tags(), [])

    def test_compute_dynamic_data(self) -> None:
        """Test compute_dynamic_data."""
        self.assertEqual(
            self.compute_dynamic_data(self.task),
            BaseDynamicTaskData(
                subject="stable",
                parameter_summary="stable (amd64)",
                runtime_context="amd64:None:hello,python3",
                configuration_context="amd64",
            ),
        )

    def test_compute_dynamic_data_variant(self) -> None:
        task_data = self.SAMPLE_TASK_DATA.copy()
        task_data["bootstrap_options"]["variant"] = DebootstrapVariant.BUILDD
        self.configure_task(task_data=task_data)

        self.assertEqual(
            self.compute_dynamic_data(self.task).parameter_summary,
            "stable (amd64-buildd)",
        )
